import asyncio
import time
import socket
import os

from datetime import datetime

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import GlobalVar

from virtualunits.vu_measurement_unit import VUMeasurementUnit
from virtualunits.meas_seq_generator import meas_seq_generator
from virtualunits.meas_seq_generator import TriggerSignal
from virtualunits.meas_seq_generator import OutputSignal
from virtualunits.meas_seq_generator import MeasurementChannel
from virtualunits.meas_seq_generator import IntegratorMode
from virtualunits.meas_seq_generator import AnalogControlMode

from urpc_enum.measurementparameter import MeasurementParameter
from urpc_enum.moverparameter import MoverParameter
from urpc_enum.temperaturecontrolparameter import TemperatureControlParameter

from fleming.common.firmware_util import *


meas_unit: VUMeasurementUnit = VUnits.instance.hal.measurementUnit

OUTPUTSIGNAL_ALPHA_HTS = 1 << 21


async def hts_alpha_laser_scan(alc_start=0.0, alc_stop=1.0, alc_step=0.01, average=10, comment=''):
    """
    Scan the HTS alpha laser current using the reference photodiode. With laser currents in the range 0.0 to 1.0.

    alc_start: Start laser current for the scan.
    alc_stop: Stop laser current for the scan. It is included in the scan range.
    alc_step: Laser current step for the scan.
    average: The number of measurements to average.
    comment: Optional comment for the header of the test report.
    """
    alc_start = float(alc_start) if (alc_start != '') else 0.0
    alc_stop = float(alc_stop) if (alc_stop != '') else 1.0
    alc_step = float(alc_step) if (alc_step != '') else 0.01
    average = int(average) if (average != '') else 10

    GlobalVar.set_stop_gc(False)

    # Delay after enabling the alpha laser power supply
    ENABLE_DELAY_S = 1.0
    # Delay after setting an alpha laser current step and before starting the excitation 
    ALC_DELAY_S = 0.1
    # Excitation time of the alpha laser in the range 0.01 to 671088.64 µs
    EXCITATION_TIME_US = 1000.0
    # Target sample rate for averaging in the range 1.5 to 2300 Hz (sample rate might be lower if excitation time is too long)
    AVG_SMAPLE_RATE_HZ = 250.0
    # Scaling factor for high range analog results
    HIGH_RANGE_SCALING = 101.0
    
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter a comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            return f"hts_alpha_temp_test() canceled"

    await send_gc_msg(f"Start EEF firmware")

    await start_firmware('eef')

    await send_gc_msg(f"Enable laser power supply")

    eef = get_node_endpoint('eef')
    meas = get_measurement_endpoint()

    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, alc_start)
    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 1)

    await asyncio.sleep(ENABLE_DELAY_S)
    
    full_reset_delay  = 40000   # 400 us
    conversion_delay  = 1200    #  12 us
    switch_delay      = 25      # 250 ns
    fixed_range       = 2000    #  20 us

    excitation_time = round(EXCITATION_TIME_US * 100)
    loop_delay = round(1.0 / AVG_SMAPLE_RATE_HZ * 1e8) - full_reset_delay - switch_delay - excitation_time - fixed_range - conversion_delay

    op_id = 'hts_alpha_laser_scan'
    seq_gen = meas_seq_generator()
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=0)   # low_result
    seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr=1)   # high_result
    seq_gen.Loop(average)
    # Reset the offsets
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
    # Enable full reset
    seq_gen.TimerWaitAndRestart(full_reset_delay - conversion_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
    # Measure the high range offset
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    # Start the integrator in low range integration
    seq_gen.TimerWaitAndRestart(switch_delay)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.low_range_reset)
    # Load the measured high range offset into the register
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
    # Measure the low range offset, then switch to auto range integration
    seq_gen.TimerWaitAndRestart(excitation_time)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_autorange)
    # Enable Alpha Laser
    seq_gen.SetSignals(OUTPUTSIGNAL_ALPHA_HTS)
    # Diable Alpha Laser, then switch to fixed range integration
    seq_gen.TimerWaitAndRestart(fixed_range)
    seq_gen.ResetSignals(OUTPUTSIGNAL_ALPHA_HTS)
    seq_gen.SetIntegratorMode(ref=IntegratorMode.integrate_with_fixed_range)
    # Load the measured low range offset into the register
    seq_gen.SetAnalogControl(ref=AnalogControlMode.read_offset)
    # Measure the result
    seq_gen.TimerWaitAndRestart(conversion_delay)
    seq_gen.SetTriggerOutput(TriggerSignal.SampleRef)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=False, addResult=True, dword=False, addrPos=0, resultPos=0)
    seq_gen.GetAnalogResult(MeasurementChannel.REF, isRelativeAddr=False, ignoreRange=False, isHiRange=True,  addResult=True, dword=False, addrPos=0, resultPos=1)
    if (loop_delay > 0):
        seq_gen.TimerWaitAndRestart(loop_delay)
    seq_gen.LoopEnd()
    # Enable full reset
    seq_gen.SetIntegratorMode(ref=IntegratorMode.full_reset)
    # Reset the offsets (only needed in case a sequence without offset correction is called after this)
    seq_gen.SetAnalogControl(ref=AnalogControlMode.full_offset_reset)
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    meas_unit.resultAddresses[op_id] = range(0, 2)
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    
    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/hts_alpha_laser_scan__{instrument}_{timestamp}.csv", 'w') as file:
        file.write(f"hts_alpha_laser_scan(alc_start={alc_start:.3f}, alc_stop={alc_stop:.3f}, alc_step={alc_step:.3f}, average={average:d}) on {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"current ; signal      ; analog_low  ; analog_high\n")

        alc_range = [alc / 1e6 for alc in range(round(alc_start * 1e6), round(alc_stop * 1e6 + 1), round(alc_step * 1e6))]
        for alc in alc_range:
            if GlobalVar.get_stop_gc():
                break
            await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, alc)
            await asyncio.sleep(ALC_DELAY_S)
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)
            analog_low = round(results[0] / average)
            analog_high = round(results[1] / average)
            signal = round(analog_low + analog_high * HIGH_RANGE_SCALING)
            file.write(f"{alc:7.3f} ; {signal:11d} ; {analog_low:11d} ; {analog_high:11d}\n")
            await send_gc_msg(f"current: {alc:.3f} ; signal: {signal:7d} ; analog_low: {analog_low:5d} ; analog_high: {analog_high:5d}")

    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, 0.0)
    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)

    return f"hts_alpha_laser_test() done"


async def hts_alpha_temp_test(duration_s=60.0, laser_current=0.0, comment=''):
    """
    Test the HTS alpha laser temperature measurement and temperature error.

    duration_s: The test duration in seconds.
    laser_current: The laser current. In the range 0.0 to 1.0.
    comment: Optional comment for the header of the test report.
    """
    duration_s = float(duration_s) if (duration_s != '') else 60.0
    laser_current = float(laser_current) if (laser_current != '') else 0.0

    GlobalVar.set_stop_gc(False)

    # Delay after enabling the alpha laser power supply
    ENABLE_DELAY_S = 1.0
    # Delay between each measurement
    SAMPLE_DELAY_S = 1.0
    
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter a comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            return f"hts_alpha_temp_test() canceled"

    await send_gc_msg(f"Start EEF firmware")

    await start_firmware('eef')

    await send_gc_msg(f"Enable laser power supply")

    eef = get_node_endpoint('eef')
    meas = get_measurement_endpoint()

    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, laser_current)
    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 1)

    await asyncio.sleep(ENABLE_DELAY_S)

    op_id = 'hts_alpha_on'
    seq_gen = meas_seq_generator()
    seq_gen.SetSignals(OUTPUTSIGNAL_ALPHA_HTS)
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)
    
    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/hts_alpha_temp_test__{instrument}_{timestamp}.csv", 'w') as file:
        file.write(f"hts_alpha_temp_test(duration_s={duration_s}, laser_current={laser_current:.3f}) on {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"time [s]  ; pd [FS]   ; temp [FS] ; error\n")

        start = time.perf_counter()
        timestamp = 0.0
        while (timestamp < duration_s):
            if GlobalVar.get_stop_gc():
                break
            results = await asyncio.gather(
                eef.GetAnalogInput(EEFAnalogInput.HTSALPHAPHOTODIODE),
                eef.GetAnalogInput(EEFAnalogInput.HTSALPHATEMPIN),
                eef.GetDigitalInput(EEFDigitalInput.HTSALPHATEMPERROR),
                asyncio.sleep(SAMPLE_DELAY_S)
            )
            pd    = results[0][0]
            temp  = results[1][0]
            error = results[2][0]
            file.write(f"{timestamp:8.3f} ; {pd:8.3f} ; {temp:8.3f} ; {error:8d}\n")
            await send_gc_msg(f"time: {timestamp:8.3f} ; pd: {pd:.3f} ; temp: {temp:.3f} ; error: {error}")
            timestamp = time.perf_counter() - start

    op_id = 'hts_alpha_off'
    seq_gen = meas_seq_generator()
    seq_gen.ResetSignals(OUTPUTSIGNAL_ALPHA_HTS)
    seq_gen.Stop(0)
    meas_unit.ClearOperations()
    await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
    await meas_unit.ExecuteMeasurement(op_id)

    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, 0.0)
    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, 0)
    
    return f"hts_alpha_temp_test() done"


async def hts_alpha_tec_test(duration_s=60.0, tec_power=1.0, comment=''):
    """
    Test the HTS alpha laser peltier element.

    duration_s: The duration of the test in seconds.
    tec_power: The TEC power to use. In the range 1.0 to 0.0 with 1.0 being min power.
    """
    duration_s = float(duration_s) if (duration_s != '') else 60.0
    tec_power = float(tec_power) if (tec_power != '') else 1.0

    GlobalVar.set_stop_gc(False)

    # Delay after enabling the TEC power supply and before checking TEC-Good
    ENABLE_DELAY_S = 1.0
    # Delay between each measurement
    SAMPLE_DELAY_S = 1.0
    
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter a comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            return f"hts_alpha_tec_test() canceled"

    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')

    await send_gc_msg(f"Enable laser power supply")

    eef = get_node_endpoint('eef')

    await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, tec_power)
    await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 1)

    await asyncio.sleep(ENABLE_DELAY_S)

    if not (await eef.GetDigitalInput(EEFDigitalInput.HTSALPHATECGOOD))[0]:
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, 1.0)
        await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 0)
        return f"hts_alpha_tec_test() failed to enable tec power supply"
    
    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/hts_alpha_tec_test__{instrument}_{timestamp}.csv", 'w') as file:
        file.write(f"hts_alpha_tec_test(duration_s={duration_s}, tec_power={tec_power:.3f}) on {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"time [s]  ; temp [FS] ; error\n")

        start = time.perf_counter()
        timestamp = 0.0
        while (timestamp < duration_s):
            if GlobalVar.get_stop_gc():
                break
            results = await asyncio.gather(
                eef.GetAnalogInput(EEFAnalogInput.HTSALPHATEMPIN),
                eef.GetDigitalInput(EEFDigitalInput.HTSALPHATEMPERROR),
                asyncio.sleep(SAMPLE_DELAY_S)
            )
            temp  = results[0][0]
            error = results[1][0]
            file.write(f"{timestamp:8.3f} ; {temp:8.3f} ; {error:8d}\n")
            await send_gc_msg(f"time: {timestamp:8.3f} ; temp: {temp:.3f} ; error: {error}")
            timestamp = time.perf_counter() - start

    await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, 1.0)
    await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 0)
    
    return f"hts_alpha_tec_test() done"


async def hts_alpha_pid_test(duration_s=300, reference_value=0.5, kp=-1.0, ti=float('inf'), td=0.0, comment=''):

    duration_s = int(duration_s) if (duration_s != '') else 300
    reference_value = float(reference_value) if (reference_value != '') else 0.5
    kp = float(kp) if (kp != '') else 1.0
    ti = float(ti) if (ti != '') else float('inf')
    td = float(td) if (td != '') else 0.0

    GlobalVar.set_stop_gc(False)
    
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter a comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            return f"hts_alpha_pid_test() canceled"

    await send_gc_msg(f"Start EEF Firmware")
    await start_firmware('eef')
    
    eef = get_node_endpoint('eef')
    await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, 1.0)
    await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 1)

    await send_gc_msg(f"Configure PID controller")
    eef_tc = get_temperature_control_endpoint('eef_tc')
    channel = 0

    await eef_tc.Enable(channel, enable=0)
    await eef_tc.ResetAnalogInputs(channel)
    await eef_tc.ResetAnalogOutputs(channel)
    await eef_tc.ResetLimiterInputs(channel)

    await eef_tc.Configure(channel, dt=1.0, kp=kp, ti=ti, td=td, min=0.0, max=0.5)

    await eef_tc.SelectAnalogInput(channel, EEFAnalogInput.HTSALPHATEMPIN, 1.0, 0.0)
    await eef_tc.SelectAnalogOutput(channel, EEFAnalogOutput.HTSALPHATEC, -1.0, 1.0)

    await eef_tc.SetParameter(channel, TemperatureControlParameter.ReferenceValue, reference_value)
    await eef_tc.SetParameter(channel, TemperatureControlParameter.DisableOutputValue, 0.0)
    await eef_tc.Enable(channel, enable=1)
    
    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/hts_alpha_pid_test__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"hts_alpha_pid_test(duration_s={duration_s}, reference_value={reference_value}, kp={kp}, ti={ti}, td={td}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"comment: {comment}")
        await write_gc_msg(file, f"time [s] ; feedback [°C] ; output [FS]")

        start = time.perf_counter()
        for timestamp in range(1, (duration_s + 1)):
            if GlobalVar.get_stop_gc():
                break

            await asyncio.sleep(start + timestamp - time.perf_counter())
            feedback = (await eef_tc.GetFeedbackValue(channel))[0]
            output = (await eef_tc.GetOutputValue(channel))[0]
            await write_gc_msg(file, f"{timestamp:8d} ; {feedback:13.1f} ; {output:11.1%}")

    await eef_tc.Enable(channel, enable=0)
    await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, 0)

    return f"hts_alpha_pid_test() done"


async def usfm_aperture_scan(usfm_start=-2.6, usfm_stop=5.0, usfm_step=0.1, comment=''):
    """
    Scan the US-LUM Aperture Presence Sensor.

    usfm_start: Start position of the US-LUM Focus Mover.
    usfm_stop: Stop position of the US-LUM Focus Mover. It is included in the scan range.
    usfm_step: Position step for the scan.
    comment: Optional comment for the header of the test report.
    """
    usfm_start = float(usfm_start) if (usfm_start != '') else -2.6
    usfm_stop = float(usfm_stop) if (usfm_stop != '') else 5.0
    usfm_step = float(usfm_step) if (usfm_step != '') else 0.1

    GlobalVar.set_stop_gc(False)
    
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter a comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            return f"hts_alpha_tec_test() canceled"

    await send_gc_msg(f"Start MC6 firmware")
    await start_firmware('mc6')

    mc6 = get_node_endpoint('mc6')
    usfm = get_mover_endpoint('usfm')

    await send_gc_msg(f"Initialize the US-LUM Focus Mover")

    await usfm.SetProfile(
        handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=28, holdPower=14, drivePowerHoldTime=1000, drivePowerFallTime=1000, timeout=1
    )
    await usfm.UseProfile(0, timeout=1)
    await usfm.SetParameter(MoverParameter.HomeSearchDirection,               0)
    await usfm.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
    await usfm.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
    await usfm.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
    await usfm.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
    await usfm.SetParameter(MoverParameter.HomePosition,             0x7FFFFFFF)
    await usfm.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
    await usfm.SetParameter(MoverParameter.MovementDirection,                 0)
    await usfm.SetConfigurationStatus(1)

    await usfm.Home()
    
    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/usfm_aperture_scan__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"usfm_aperture_scan(usfm_start={usfm_start:.3f}, usfm_stop={usfm_stop:.3f}, usfm_step={usfm_step:.3f}) on {instrument} started at {timestamp}")
        await write_gc_msg(file, f"comment: {comment}")
        await write_gc_msg(file, f"pos [mm] ; aperture")

        pos_range = [pos / 1e6 for pos in range(round(usfm_start * 1e6), round((usfm_stop + usfm_step) * 1e6), round(usfm_step * 1e6))]
        for pos in pos_range:
            if GlobalVar.get_stop_gc() is True:
                await usfm.Home()
                return f"usfm_aperture_scan() stopped by user"
            
            await usfm.Move(round(pos / 0.0127 * 256))
            await asyncio.sleep(0.1)
            if ((await mc6.GetDigitalInput(MC6DigitalInput.REF8))[0] == 0):
                aperture = 1
            else:
                aperture = 0

            await write_gc_msg(file, f"{pos:8.3} ; {aperture}")

        pos_error = (await usfm.Home())[0]
        await write_gc_msg(file, f"\nPosition Error: {pos_error}")

    return f"usfm_aperture_scan() done"


async def hts_alpha_set_tec(tec_power=1.0, enable=1):

    tec_power = float(tec_power) if (tec_power != '') else 1.0
    enable = int(enable) if (enable != '') else 1

    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')

    await send_gc_msg(f"Set TEC power: {tec_power}, enable: {enable}")

    eef = get_node_endpoint('eef')

    await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC, tec_power)
    await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE, enable)
    
    return f"hts_alpha_set_tec() done"


async def hts_alpha_set_laser(laser_current=0.0, enable=1, excitation=0):

    laser_current = float(laser_current) if (laser_current != '') else 0.0
    enable = int(enable) if (enable != '') else 1
    excitation = int(excitation) if (excitation != '') else 0
    
    await send_gc_msg(f"Start EEF firmware")
    await start_firmware('eef')

    await send_gc_msg(f"Set laser current: {laser_current}, enable: {enable}, excitation: {excitation}")

    meas = get_measurement_endpoint()

    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserPower, laser_current)
    await meas.SetParameter(MeasurementParameter.HTSAlphaLaserEnable, enable)

    if excitation:
        op_id = 'hts_alpha_on'
        seq_gen = meas_seq_generator()
        seq_gen.SetSignals(OUTPUTSIGNAL_ALPHA_HTS)
        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
    else:
        op_id = 'hts_alpha_off'
        seq_gen = meas_seq_generator()
        seq_gen.ResetSignals(OUTPUTSIGNAL_ALPHA_HTS)
        seq_gen.Stop(0)
        meas_unit.ClearOperations()
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)
        await meas_unit.ExecuteMeasurement(op_id)
    
    return f"hts_alpha_set_laser() done"

async def us_lum_test(window_ms=1000.0, iterations=10, comment=''):

    window_ms = float(window_ms) if (window_ms != '') else 1000.0
    iterations = int(iterations) if (iterations != '') else 100
    
    GlobalVar.set_stop_gc(False)

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/us_lum_test__{instrument}_{timestamp}.csv", 'w') as file:
        await write_gc_msg(file, f"us_lum_test(window_ms={window_ms}, iterations={iterations}, comment={comment}) on {instrument} started at {timestamp}")

        window_us = round(window_ms * 1000)
        window_coarse, window_fine = divmod(window_us, 65536)

        pre_cnt_window = 100    # 1 us

        op_id = 'us_lum_test'
        seq_gen = meas_seq_generator()

        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 0)  # pmt3_cr_lsb
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 1)  # pmt3_cr_msb
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 2)  # pmt3_dt_lsb
        seq_gen.ClearResultBuffer(relative=False, dword=False, addrReg=0, addr= 3)  # pmt3_dt_msb

        seq_gen.TimerWaitAndRestart(pre_cnt_window)
        seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=False, resetCounter=True, resetPresetCounter=True, correctionOn=False)
        if window_coarse > 0:
            seq_gen.Loop(window_coarse)
            seq_gen.Loop(65536)
            seq_gen.TimerWaitAndRestart(pre_cnt_window)
            seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.LoopEnd()
            seq_gen.LoopEnd()
        if window_fine > 0:
            seq_gen.Loop(window_fine)
            seq_gen.TimerWaitAndRestart(pre_cnt_window)
            seq_gen.PulseCounterControl(MeasurementChannel.US_LUM, cumulative=True, resetCounter=False, resetPresetCounter=True, correctionOn=True)
            seq_gen.LoopEnd()

        seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=False, relative=False, resetCounter=False, cumulative=True, dword=True, addrPos=0, resultPos=0)
        seq_gen.GetPulseCounterResult(MeasurementChannel.US_LUM, deadTime=True, relative=False, resetCounter=True, cumulative=True, dword=True, addrPos=0, resultPos=2)
        seq_gen.Stop(0)

        meas_unit.ClearOperations()
        meas_unit.resultAddresses[op_id] = range(0, 14)
        await meas_unit.LoadTriggerSequence(op_id, seq_gen.currSequence)

        pmt3_cr_sum = 0
        pmt3_dt_sum = 0
        pmt3_t_sum  = 0

        await write_gc_msg(file, f"iteration ; pmt3_cr     ; pmt3_dt     ; pmt3_t [ns]")

        for i in range(iterations):
            if GlobalVar.get_stop_gc() is True:
                return f"us_lum_test() stopped by user"
            
            await meas_unit.ExecuteMeasurement(op_id)
            results = await meas_unit.ReadMeasurementValues(op_id)

            pmt3_cr = results[0] + (results[1] << 32)
            pmt3_dt = results[2] + (results[3] << 32)

            pmt3_t = (pmt3_dt / 1.2e9 / pmt3_cr) * 1e9 if (pmt3_cr > 0) else 0.0

            pmt3_cr_sum += pmt3_cr
            pmt3_dt_sum += pmt3_dt
            pmt3_t_sum  += pmt3_t

            await write_gc_msg(file, f"{(i + 1):9d} ; {pmt3_cr:11d} ; {pmt3_dt:11d} ; {pmt3_t:11.6f}")

        pmt3_cr_mean = pmt3_cr_sum / iterations
        pmt3_dt_mean = pmt3_dt_sum / iterations
        pmt3_t_mean  = pmt3_t_sum / iterations

        await write_gc_msg(file, f"")
        await write_gc_msg(file, f"  Average ; {pmt3_cr_mean:11.0f} ; {pmt3_dt_mean:11.0f} ; {pmt3_t_mean:11.6f}")

    return f"us_lum_test() done"

async def USL_PCB_check(comment=''):

    eef = get_node_endpoint('eef')
    TestLoop = False

    await start_firmware('eef')

    async def NTC_BUTTON_CHECK():
        NTCLevel = 1.0
        ButtonDelay = 0.1
        print("..... Waiting for the NTC button")
        KeyVal = await eef.GetAnalogInput(14,1)
        NTCLevel = KeyVal[0]
        
        while (NTCLevel >= 0.1):                           # Warte auf Knopf drücken
            await asyncio.sleep(ButtonDelay)
            KeyVal = await eef.GetAnalogInput(14,1)
            NTCLevel = KeyVal[0]

        while (NTCLevel <= 0.1):                           # Warte auf Knopf loslassen
            await asyncio.sleep(ButtonDelay)
            KeyVal = await eef.GetAnalogInput(14,1)
            NTCLevel = KeyVal[0]

    print("\n")
    if (comment == '') and not is_ground_control():
        comment = input(f"Enter USL-Number and comment ('X' to cancel): ")
        if comment == 'X' or comment == 'x':
            print("\n")
            return f"USL_PCB_check() canceled"

    instrument = socket.gethostname()
    timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
    report_dir = f"{os.path.dirname(__file__)}/usl_test_results"
    os.makedirs(report_dir, exist_ok=True)
    with open(f"{report_dir}/USL_PCB_check_{instrument}_{timestamp}.csv", 'w') as file:
        file.write(f"USL_PCB_check with {instrument} started at {timestamp}\n")
        file.write(f"comment: {comment}\n")
        file.write(f"\n")

        # LASER SUPPLY CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        await eef.SetDigitalOutput(1,1,1)
        print("Check the Voltage and Current Values of the Power Supply!")
        print("\n")
        print("If OKAY -> Press [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        file.write(f"The power consumption of the USLUM board is in the right range")
        file.write(f"\n")
        print("\n")

        # LASER CURRENT CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        print("Connect the Amperemeter at the sockets <Laser current>")
        print("Current will be flow for 5 seconds")
        print("Current must be between 0,873 to 0,927A -> Press [NTC]-Button to start")
        await NTC_BUTTON_CHECK()    
        file.write(f"Laser current check: ")
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHACURRENT,1,1)
        await seq_set_signal('htsalpha')
        await asyncio.sleep(5)
        await seq_clear_signal('htsalpha')
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHACURRENT,0,1)
        print("\n")
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        file.write(f" OKAY ,\n")
        print("\n")

        # LASER CURRENT CUTOFF CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        print("Laser Current will be activatet for 10 seconds and while flowing,")
        print("must be able to stop with the NTC button. -> Press [NTC]-Button to start")
        await NTC_BUTTON_CHECK()
        file.write(f"Laser current CUTOFF check: ")
        print("\n")
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHACURRENT,1,1)
        await seq_set_signal('htsalpha')
        await asyncio.sleep(10)
        await seq_clear_signal('htsalpha')
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHACURRENT,0,1)
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        file.write(f" OKAY \n")
        print("\n")

        # PHOTODIODE FEEDBACK CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        print("Checking the laser feedback signal from the photodiode -> Press [NTC]-Button to start")
        await NTC_BUTTON_CHECK()
        file.write(f"Photodiode Feedback check:\n")
        Value = await eef.GetAnalogInput(EEFAnalogInput.HTSALPHAPHOTODIODE,1)
        print("\n")
        print("... unpressed <PD>-Button =",Value[0])
        file.write(f"... unpressed <PD>-Buttons: {Value[0]}\n")
        print("\n")
        print(" If OKAY -> Press <1.PD>-Button and continue with [NTC]-Button")
        await NTC_BUTTON_CHECK()
        Value = await eef.GetAnalogInput(EEFAnalogInput.HTSALPHAPHOTODIODE,1)
        print("\n")
        print("........... <1.PD>-Button =",Value[0])
        file.write(f"............ <1.PD>-Button: {Value[0]}\n")
        print("\n")
        print(" If OKAY -> Press <2.PD>-Button amd continue with [NTC]-Button")
        await NTC_BUTTON_CHECK()
        Value = await eef.GetAnalogInput(EEFAnalogInput.HTSALPHAPHOTODIODE,1)
        print("\n")
        print("........... <2.PD>-Button =",Value[0])
        file.write(f"............ <2.PD>-Button: {Value[0]}\n")
        print("\n")
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        print("\n")
 
        #HIGH VOLTAGE CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        print("Connect the Voltmeter at the sockets <HV-Voltage>")
        print("HV-Power 1500V will be activated for 5 seconds")
        print("Voltage must be between 147,5 to 151,5V -> Press [NTC]-Button to start")
        await NTC_BUTTON_CHECK()
        file.write(f"High Voltage check:\n")
        await eef.SetAnalogOutput(EEFAnalogOutput.PMTUSLUMHIGHVOLTAGESETTING,1,1)
        await eef.SetDigitalOutput(EEFDigitalOutput.PMT3HIGHVOLTAGEENABLE,0,1)
        await asyncio.sleep(2.5)
        Value = await eef.GetAnalogInput(EEFAnalogInput.PMTUSLUMHIGHVOLTAGEMONITOR,1)
        print("HV Voltage monitor:",Value[0])
        print("HV Voltage calculated:",Value[0]*1500)
        file.write(f"Monitoring value: {Value[0]}\n")       
        await asyncio.sleep(2.5)
        await eef.SetDigitalOutput(EEFDigitalOutput.PMT3HIGHVOLTAGEENABLE,1,1)
        await eef.SetAnalogOutput(EEFAnalogOutput.PMTUSLUMHIGHVOLTAGESETTING,0,1)
        print("\n")
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        print("\n")

        #PELTIER POWER CHECK ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        print("Connect the Voltmeter and Amperemeter at the TEC sockets")
        print("TEC Power will be activated for 10 seconds")
        print("The Current  must be between 3,0 to 3,4A")
        print("The Voltage  must be between 7,4 to 8,0V -> Press [NTC]-Button to start")
        await NTC_BUTTON_CHECK()
        file.write(f"Peltier Power check: ")
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC,0,1)
        await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE,1,1)
        await asyncio.sleep(10)
        await eef.SetDigitalOutput(EEFDigitalOutput.HTSALPHATECENABLE,0,1)
        await eef.SetAnalogOutput(EEFAnalogOutput.HTSALPHATEC,1,1)
        print("\n")
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        file.write(f" OKAY \n")
        TestLoop = True;
        file.close()

        #DISCRIMINATOR LEVEL CHECK with INTERNAL 250MHZ CLOCK ~~~~~~~~~~~~~~~~~~~~~~~~~~~
    if TestLoop:
        print("\n")
        print("Discriminator Scan Check - Check the clock connection to the USL-Testboard")
        print("If OKAY -> Continue with [NTC]-Button -> else Power off and check the hardware")
        await NTC_BUTTON_CHECK()
        await pmt3_dscan_v7of(0.1,0.7,0.001,100,file.name)
        return f"USL_PCB_check() done"
    else:
        return f"Test run ended with error"
